from flask import Flask
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from .models import Base
from .config import Config
from flask_jwt_extended import JWTManager
import pandas as pd
import urllib
from flask_cors import CORS, cross_origin

def create_app():
    app = Flask(__name__)
    app.config.from_object(Config)  # Load configuration from Config class
    app.config['JWT_SECRET_KEY'] = 'your_secret_key'
    CORS(app, support_credentials=True)

    # Initialize JWT
    jwt = JWTManager(app)

    # Determine the database type
    db_type = app.config['DB_TYPE']  # Get the database type from config

    # Create the appropriate database connection
    if db_type == 'mysql':
        engine = create_engine(app.config['MYSQL_DATABASE_URI'])
    elif db_type == 'oracle':
        engine = create_engine(app.config['ORACLE_DATABASE_URI'])
    elif db_type == 'msaccess':
        # Prepare the ODBC connection string
        msaccess_uri = app.config['MSACCESS_DATABASE_URI']
        encoded_uri = urllib.parse.quote(msaccess_uri, safe='')
        engine = create_engine(f"access+pyodbc:///?odbc_connect={encoded_uri}")
    elif db_type == 'excel':
        # Note: Excel files are not typically handled through SQLAlchemy
        # Load Excel data with pandas
        app.excel_data = pd.read_excel(app.config['EXCEL_FILE_PATH'])
    else:  # Default to SQLite
        engine = create_engine(app.config['SQLITE_DATABASE_URI'])

    app.engine = engine

    # Create all tables in the database if applicable
    if db_type not in ['excel']:  # Skip table creation for Excel
        Base.metadata.create_all(engine)

    # Register blueprints
    from .routes import main
    app.register_blueprint(main)

    return app
